***************************************************************
* Simulation
***************************************************************

**************************************************************
* Read in data
**************************************************************

set more off
clear
* Put name of your main directory path here:
local main = "___"
cd `main'
use data_monthly.dta, clear

***************************************************************
* Set color scheme (optional)
***************************************************************

set scheme s1color

***************************************************************
* Data work
***************************************************************
 
* Relative price fluctuations (supply fluctuations)
g z			= CPI - Median
* 12-month average
gen z_12ma	= (1/12)*(z + z[_n-1] + z[_n-2] + z[_n-3] + z[_n-4] + z[_n-5] + z[_n-6] + z[_n-7] + z[_n-8] + z[_n-9] + z[_n-10] + z[_n-11])
* Lag
g L1z		= z[_n-1]

* Dates and sample
local latest = tm(2022m9)	
local latestbc = tm(2022m8)	
local latestpe = tm(2022m9)

* Sample dummies
forval s = 1/4 {
	g sample`s' = 0
}
replace sample1 = 1 if (t>=tm(1960m1)) & (t<=tm(1972m12))
replace sample2 = 1 if (t>=tm(1973m1)) & (t<=tm(1984m12))
replace sample3 = 1 if (t>=tm(1985m1)) & (t<=tm(2019m12))
replace sample4 = 1 if (t>=tm(2020m1)) & (t<=`latest')
	
***************************************************************
* INPUT SIMULATION ASSUMPTIONS
***************************************************************

* Shock lowers V/U to level without fiscal stimulus.
* Shock to V/U calibrated based on Barnichon and others (2021).
* https://www.frbsf.org/economic-research/publications/economic-letter/2021/october/is-american-rescue-plan-taking-us-back-to-1960s/
* -0.6pp by end-2021 and -0.5pp by end-2021.  Assume shock fades to zero by end-2023 (although that is beyond our sample of interest).

* Use 
g shock			=  0	if t<=tm(2021m3)
replace shock	= -0.6	if t==tm(2021m12)
replace shock	= -0.5	if t==tm(2022m12)
replace shock	=  0.0	if t==tm(2023m12)
* Interpolate ("int") between missing datapoints
ipolate shock t if t>=tm(2021m3), generate(shockint)
replace shock = shockint if t>=tm(2021m4) & shock==.

* Simulated ("s") path of V/U
* Set "shockon" = 1 if shock is on; =0 if shock is off.
local shockon = 1
g svu		= vu if t<=tm(2021m3)
replace svu	= vu + `shockon'*shock if t>=tm(2021m4)	
	
* Expectations anchoring
* Assumed feedback from inflation to change in expected inflation
* Note: feedback = 0 is fully anchored; feedback = 1 is backward-looking. Empirics: 0.01.
local feedback	= 0
* Can also assume exepectations replicate exactly what they actually did.
* But then need to set denachoring feedback to zero (feedback = 0).
* Replication: off (replicate = 0); on (replicate = 1).
local replicate	= 1

* Simulated "z" shocks for t>=tm(2021m4)
* Use actual "z" shocks
g zsim = z if t>=tm(2021m3)

***************************************************************
* Variables
***************************************************************
	
* Vacancy rate
* V/U and 12-month average
g lu	= ln(u)
g lv	= ln(v)
label variable lv "Log vacancy rate"
label variable lu "Log unemployment rate"

* Estimate BC relation and save coefficients and fitted values
* Relation between lv and lu
reg lv lu if t>=tm(2020m4) & t<=`latestbc', r
local a = _b[_cons]
local b = _b[lu]
predict lvhat
g vhat = exp(lvhat)

* Obtain simulated u ("su") using estimated BC relation and simulated vu ("svu") path
* Assume the observed residuals ("eps") from the estimated BC curve ("lvresidual") also apply in the simulation
g eps = lv - lvhat
* Derive log of su based on estimated V/U coefficients and solution for U after subtracting ln(U) from both sides to have
* equation with ln(V/U) on LHS with ln(U) on RHS.
g lsu	= ( 1/(`b'-1) )*(ln(svu) - `a' - eps)
g su	= exp(lsu)
* Compute simulated V ("sv")
g sv	= svu*su

* 12-month "simulated" average of V/U
g svu_12ma	= (1/12)*(svu + svu[_n-1] + svu[_n-2] + svu[_n-3] + svu[_n-4] + svu[_n-5] + svu[_n-6] + svu[_n-7] + svu[_n-8] + svu[_n-9] + svu[_n-10] + svu[_n-11]) 

* Simulated "z" terms ("sz")
g sz		= z if t<=tm(2021m3)
replace sz 	= zsim if t>=tm(2021m4)
* 12-month average and additional terms
g sz_12ma	= (1/12)*(sz + sz[_n-1] + sz[_n-2] + sz[_n-3] + sz[_n-4] + sz[_n-5] + sz[_n-6] + sz[_n-7] + sz[_n-8] + sz[_n-9] + sz[_n-10] + sz[_n-11])
g sz_12ma2	= sz_12ma^2
g sz_12ma3	= sz_12ma^3

***************************************************************
* Simulate variables
***************************************************************

* Headline inflation
g headline = CPI if t<=`latest'

* Phillips curve estimation
* Define PC variables:
cap drop y x*
g y		= Median-pe
* Measure of slack
foreach var in vu {
	* Define measure of slack ("x")
	cap drop x*
	g x		= `var'_12ma
	g x2	= x^2
	g x3	= x^3
	* Compute nonlinear "z" terms
	g z_12ma2 = z_12ma^2
	g z_12ma3 = z_12ma^3	
	* Estimation
	reg y x x2 x3 z_12ma z_12ma2 z_12ma3 if (sample3==1|sample4==1), r
}

* Predict inflation gap
predict yhat
* Residuals
g residual = y - yhat

* Predict inflation gap based on simulated "x" and "z" terms
* Simulated x terms:
g sx	= svu_12ma
g sx2	= sx^2
g sx3	= sx^3

* Call this simulated yhat ("syhat")
g syhat = yhat if t<=tm(2021m3)
replace syhat = _b[_cons] + _b[x]*sx + _b[x2]*sx2 + _b[x3]*sx3 + _b[z_12ma]*sz_12ma + _b[z_12ma2]*sz_12ma2 + _b[z_12ma3]*sz_12ma3 if t>=tm(2021m4)
* Add the residual, assuming that also applies in the counterfactual scenario
replace syhat = syhat + residual 

* Insample predict inflation rate
g phat = yhat + pe if e(sample)

* Compute simulated ("s") inflation ("p"), change ("d") in expected inflation, and expected inflation
g sp	= Median		if t<=tm(2021m3)
g spe	= pe			if t<=tm(2021m3)
g dspe	= pe - pe[_n-1]	if t<=tm(2021m3)
* This requires loop over future dates
replace dspe = 0 if t==tm(2021m3)+1
forval h = 1/50 {
	replace spe	= spe[_n-1] + dspe							if t==tm(2021m3)+`h'
	replace sp	= syhat + spe								if t==tm(2021m3)+`h'
	replace headline = sp + sz								if t==tm(2021m3)+`h'
	replace dspe = `feedback'*(headline[_n-1] - spe[_n-1])	///
	+ `replicate'*(pe - pe[_n-1])							if t==tm(2021m3)+`h'+1	
}	

* Compute 12-month rates ("a" for annual)
foreach var in sp {
	gen index_`var'		= 100 if t==tm(1989m12)
	replace index_`var'	= index_`var'[_n-1]*(1 + (((1+(`var'/100))^(1/12))-1)) if t>tm(1989m12)
	gen `var'a			= 100*(((index_`var'/index_`var'[_n-12]))-1)	
}

* Name variables
foreach var in vu u Median {
	label variable `var' "Actual"
}
foreach var in svu su sp {
	label variable `var' "Counterfactual"
}

***************************************************************
* Plot results
***************************************************************

g plotsample = 1 if t>=tm(2021m1) & t<=`latest'

local title = "V/U (monthly)"
local index = 2
preserve
keep if plotsample==1
sum t
local t0 = r(min)
local t1 = r(max)
gen y1 = vu
gen y2 = svu if t>=tm(2021m3)
gen y3 = vu_12ma
gen y4 = svu_12ma if t>=tm(2021m3)
label variable y1 "Actual"
label variable y2 "Counterfactual"
twoway (line y1 y2 t, ///
lcolor(midblue red midblue red) ///
lpattern(solid solid dash dash .-) ///
lwidth(medthick medthick medthick medthick medthick) ///
title("`title'", size(medium)) ///
xmtick(#20) xlabel(`t0'(12)`t1', format(%tmCY) angle(90)) xlabel(,labsize(medium)) ylabel(,labsize(medium) angle(0)) xtitle("") ytitle("") ///
legend(row(11) symxsize(6) ring(0) position(11) bmargin(none) region(style(none))) ///
legend(order(1 2)) ///
legend(size(small)) ysize(5) ///
xsize(6) ///
saving(`index'.gph, replace))
restore
*/

local title = "V/U (12-month average)"
local index = 21
preserve
keep if plotsample==1
sum t
local t0 = r(min)
local t1 = r(max)
gen y1 = vu
gen y2 = svu if t>=tm(2021m3)
gen y3 = vu_12ma
gen y4 = svu_12ma if t>=tm(2021m3)
label variable y3 "Actual"
label variable y4 "Counterfactual"
twoway (line y3 y4 t, ///
lcolor(midblue red midblue red) ///
lpattern(solid solid dash dash .-) ///
lwidth(medthick medthick medthick medthick medthick) ///
title("`title'", size(medium)) ///
xmtick(#20) xlabel(`t0'(12)`t1', format(%tmCY) angle(90)) xlabel(,labsize(medium)) ylabel(,labsize(medium) angle(0)) xtitle("") ytitle("") ///
legend(row(11) symxsize(6) ring(0) position(11) bmargin(none) region(style(none))) ///
legend(order(1 2)) ///
legend(size(small)) ysize(5) ///
xsize(6) ///
saving(`index'.gph, replace))
restore
*/

//*
local title = "Core Inflation (monthly; percent)"
local index = 3
preserve
keep if plotsample==1
sum t
local t0 = r(min)
local t1 = r(max)
gen y1 = Median
gen y2 = sp if t>=tm(2021m3)
gen y3 = .
gen y4 = .
gen y5 = . if t>=tm(2021m3)
label variable y1 "Actual"
label variable y2 "Counterfactual"
twoway (line y1 y2 y3 y4 y5 t, ///
lcolor(midblue red midblue midblue red) ///
lpattern(solid solid dash dot dot) ///
lwidth(medthick medthick medthick medthick medthick medthick) ///
title("`title'", size(medium)) ///
xmtick(#20) xlabel(`t0'(12)`t1', format(%tmCY) angle(90)) xlabel(,labsize(medium)) ylabel(,labsize(medium) angle(0)) xtitle("") ytitle("") ///
legend(row(11) symxsize(6) ring(0) position(11) bmargin(none) region(style(none))) ///
legend(order(1 2)) ///
legend(size(small)) ysize(5) ///
xsize(6) ///
saving(`index'.gph, replace))
restore
*/

//*
local title = "Core Inflation (12-month; percent)"
local index = 4
preserve
keep if plotsample==1
sum t
local t0 = r(min)
local t1 = r(max)
gen y1 = Median_12	
gen y2 = spa if t>=tm(2021m3)
gen y3 = .
gen y4 = .
gen y5 = . if t>=tm(2021m3)
label variable y1 "Actual"
label variable y2 "Counterfactual"
twoway (line y1 y2 y3 y4 y5 t, ///
lcolor(midblue red midblue midblue red) ///
lpattern(solid solid dash dot dot) ///
lwidth(medthick medthick medthick medthick medthick medthick) ///
title("`title'", size(medium)) ///
xmtick(#20) xlabel(`t0'(12)`t1', format(%tmCY) angle(90)) xlabel(,labsize(medium)) ylabel(,labsize(medium) angle(0)) xtitle("") ytitle("") ///
legend(row(11) symxsize(6) ring(0) position(11) bmargin(none) region(style(none))) ///
legend(order(1 2)) ///
legend(size(small)) ysize(5) ///
xsize(6) ///
saving(`index'.gph, replace))
restore
*/

************************************************
* Save chart data
************************************************

preserve
keep if plotsample==1
rename Median Core_monthly_Actual
rename Median_12 Core_12month_Actual
rename sp Core_monthly_Counterfactual
rename spa Core_12month_Counterfactual
rename vu vu_monthly_Actual
rename vu_12ma vu_12ma_Actual
rename svu vu_monthly_Counterfactual
rename svu_12ma vu_12ma_Counterfactual
outsheet Core_monthly_Actual Core_12month_Actual Core_monthly_Counterfactual Core_12month_Counterfactual vu_monthly_Actual vu_12ma_Actual vu_monthly_Counterfactual vu_12ma_Counterfactual t using "results\charts\Figure 8.xls", replace
restore

graph combine 2.gph 21.gph 3.gph 4.gph, ysize(5)
*/
